import { Router } from 'express';
import { sellersController } from './controller';
import { authenticateUser, requireAdmin } from '@/middlewares/auth';
import { checkFeature } from '@/middlewares/featureGate';

const router = Router();

// All seller management endpoints require admin/staff/supervendor role and manage_sellers feature
const sellerManagementMiddleware = [
  authenticateUser,
  requireAdmin, // This covers admin, staff, and supervendor roles
  checkFeature('manage_sellers')
];

// Create new seller
router.post('/', sellerManagementMiddleware, sellersController.createSeller);

// List all sellers for current admin/supervendor
router.get('/', sellerManagementMiddleware, sellersController.listSellers);

// Get seller profile by ID
router.get('/:id', sellerManagementMiddleware, sellersController.getSeller);

// Update seller details
router.put('/:id', sellerManagementMiddleware, sellersController.updateSeller);

// Soft delete seller (mark as inactive)
router.delete('/:id', sellerManagementMiddleware, sellersController.deleteSeller);

// Add or update seller inventory
router.post('/:id/inventory', sellerManagementMiddleware, sellersController.addInventory);

// List seller inventory
router.get('/:id/inventory', sellerManagementMiddleware, sellersController.getInventory);

// Update inventory item
router.put('/:id/inventory/:item_id', sellerManagementMiddleware, sellersController.updateInventory);

// Delete inventory item
router.delete('/:id/inventory/:item_id', sellerManagementMiddleware, sellersController.deleteInventory);

// Assign seller to shopping list item
router.post('/:id/assign-order', sellerManagementMiddleware, sellersController.assignOrder);

// Mark seller order as fulfilled
router.put('/:id/mark-fulfilled', sellerManagementMiddleware, sellersController.markFulfilled);

// Get seller analytics
router.get('/:id/analytics', sellerManagementMiddleware, sellersController.getAnalytics);

export default router; 